/**
 * Handle exporting and downloading of data to an Excel file
 * @type {{}}
 */
mods.export = function($) {

    function getStatus(tokenId, cb) {
        mods.server.get('/export/status/' + tokenId, cb);
    }

    function updateProgress(prog) {
        var width = Math.round(prog * 100) + '%';
        $('#prog-ind').width(width);
        $('#prog-label').text(width);
    }

    function showError() {
        mods.app.showMessage({
            title: 'Excel Exportation',
            message: 'An error happened while exporting to excel',
            type: 'OK'
        });
    }

    function showProgress() {
        var progModal = mods.modal.create({
            id: 'prog',
            header: 'Generating Excel File',
            content: '<div style="margin:0px 20px 10px 20px" class="progress-bar">' +
                '<div id="prog-label" class="progress-bar-label"></div>' +
                '<div class="progress-bar-box">' +
                '<div id="prog-ind" class="progress-bar-inner"></div></div></div>',
            footer: '<div style="text-align:center"><a id="btn-prog-cancel" class="btn-link">Cancel</a></div>'
        });
        progModal.show();
        updateProgress(0);

        $('#btn-prog-cancel').click(function() {
            progModal.cancel();
        });

        return progModal;
    }

    function monitor(progModal, tokenId) {
        getStatus(tokenId, function(res) {
            if (res.status === 'RUNNING') {
                updateProgress(res.progress);
                // update its status on each 0.5 seconds
                setTimeout(function() {
                    monitor(progModal, tokenId);
                }, 500);
                return;
            }

            progModal.hide();

            if (res.status === 'ERROR') {
                showError();
                return;
            }

            if (res.status === 'FINISHED') {
                mods.export.download(tokenId);
            }
        });
    }

    return {
        /**
         * Execute the exporting to Excel and initiate the download process. In case of
         * errors during the initialization process, they are passed to the callback function
         * @param url the URL path (after the context path) to start the exporting
         * @param data the data used by the URL to start the exporting, if necessary
         * @param cb the callback function called in case of error
         */
        execute: function(url, data, cb) {
            var progModal = showProgress();
            var canceled = false;

            progModal.onCancel = function() {
                canceled = true;
                progModal.hide();
            }

            mods.server.post(url, data, function(res) {
                // error or it was canceled ?
                if (!res.success || canceled) {
                    // callback function was set ?
                    if (cb) {
                        cb(res.errors);
                    }
                    return;
                }

                var tokenId = res.result;

                progModal.onCancel = function() {
                    mods.export.cancel(tokenId);
                };

                monitor(progModal, tokenId);
            },
            function(err) {
                progModal.hide();
            });
        },

        download: function(tokenId) {
            window.location.href = window.appconfig.context + '/resources/api/export/download/' + tokenId;
        },

        cancel: function(tokenId, cb) {
            mods.server.post('/export/cancel/' + tokenId, cb);
        },
    };

}(jQuery);